/*:
 * @target MZ
 * @plugindesc v1.0 初回起動で言語をデフォルト値（例: 英語=1）に固定し、必要なら変数へ一度だけ同期
 * @author HS
 *
 * @help
 * HS_LangSimple の上に手を入れず、「初回起動の段階で言語indexを既定値にする」ための
 * だけの小プラグインです。Options の表示より前に反映されるため、初回の乖離を防げます。
 *
 * 【推奨順序】HS_LangSimple → 本プラグイン →（ブリッジ/互換/ローカライズ各種）
 *
 * @param defaultIndex
 * @text 既定の言語index
 * @type number
 * @min 0
 * @default 1
 * @desc 0=日本語, 1=English, 2=简体中文, 3=한국어 … の想定。HS_LangSimpleの並びと一致させてください。
 *
 * @param mode
 * @text 適用モード
 * @type select
 * @option 既存設定がない時だけ(IF_UNSET)
 * @value IF_UNSET
 * @option 毎回上書き(ALWAYS)
 * @value ALWAYS
 * @default IF_UNSET
 * @desc 通常は IF_UNSET（初回のみ設定）。毎回英語で起動したい場合だけ ALWAYS を使ってください。
 *
 * @param languageVarId
 * @text 言語インデックス変数ID
 * @type variable
 * @default 0
 * @desc ゲーム内の言語変数。0ならHS_LangSimpleの設定から自動取得。ロード成功時にも一度だけ同期します。
 */
(() => {
  "use strict";
  const PN  = "HS_LangForceDefault";
  const PRM = PluginManager.parameters(PN);
  const DEF = Number(PRM.defaultIndex || 1);
  const MODE = String(PRM.mode || "IF_UNSET"); // "IF_UNSET" | "ALWAYS"

  // 言語変数ID（未指定なら HS_LangSimple の設定を参照）
  function resolveVarId(){
    let id = Number(PRM.languageVarId || 0);
    if (!id) {
      const p = PluginManager.parameters("HS_LangSimple") || {};
      id = Number(p.languageVarId || 0);
    }
    return id;
  }
  const VAR_ID = resolveVarId();

  // 初期適用：タイトルに行く前に実行（Configは Scene_Boot.start で読み込まれた直後の段階）
  const _bootStart = Scene_Boot.prototype.start;
  Scene_Boot.prototype.start = function(){
    // 既存設定がない or 毎回上書き の場合に既定値を適用
    const cur = (ConfigManager && typeof ConfigManager.hs_langIndex === "number")
      ? Number(ConfigManager.hs_langIndex) : NaN;

    if (MODE === "ALWAYS" || !(Number.isInteger(cur))) {
      ConfigManager.hs_langIndex = Number(DEF);
      if (ConfigManager.save) ConfigManager.save();
    }

    _bootStart.call(this);
  };

  // ロード成功時：Config → 変数 へ “一度だけ” 同期（影響を最小化）
  const _onLoadSuccess = Scene_Load.prototype.onLoadSuccess;
  Scene_Load.prototype.onLoadSuccess = function(){
    _onLoadSuccess.call(this);
    try {
      if (VAR_ID > 0 && window.$gameVariables) {
        $gameVariables.setValue(VAR_ID, Number(ConfigManager.hs_langIndex ?? DEF));
      }
    } catch(_) {}
  };

  // ニューゲーム直後（最初のマップ開始時）にも一度だけ同期
  const _mapStart = Scene_Map.prototype.start;
  Scene_Map.prototype.start = function(){
    _mapStart.call(this);
    try {
      if (VAR_ID > 0 && window.$gameVariables && window.$gameSystem && !$gameSystem._hsLangSyncedOnce){
        $gameVariables.setValue(VAR_ID, Number(ConfigManager.hs_langIndex ?? DEF));
        $gameSystem._hsLangSyncedOnce = true; // ←毎マップ書き換えない
      }
    } catch(_) {}
  };
})();
